/*
 *  Copyright (C) 2002-2007  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dosbox.h"
#include "support.h"
#include "setup.h"
#include "vga.h"
#include "inout.h"

/* By defining any of this as 0 the corresponding drivers can be completely disabled */
#ifndef C_SVGA_S3
#define C_SVGA_S3 1
#endif
#ifndef C_SVGA_TSENG
#define C_SVGA_TSENG 1
#endif
#ifndef C_SVGA_PARADISE
#define C_SVGA_PARADISE 1
#endif

SVGA_Driver svga = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };

// Forward declarations - driver initialization routines
#if C_SVGA_S3
bool VGA_SetupS3Trio(const char*);
#endif
#if C_SVGA_TSENG
bool VGA_SetupET4K(const char*);
bool VGA_SetupNewET4K(const char*);
bool VGA_SetupET3K(const char*);
#endif
#if C_SVGA_PARADISE
bool VGA_SetupPVGA1A(const char*);
#endif

// Simple setup for "none"
bool VGA_SetupSVGANone(const char* testname)
{
	if(testname)
		return strstr(testname, "none") != NULL;
	svgaCard = SVGA_None;
	vga.vmemsize = 256*1024; // Plain VGA
	vga.vmemwrap = vga.vmemsize;
	return true;
}

// Prototype for driver initialization function. If called with non-NULL parameter
// it should return true if the parameter value is valid name for the supported chipset,
// false otherwise. Called with NULL as parameter means that the chipset has to be
// initialized -- all vectors to be set, VGA setting adjusted, video modes inited, etc.
typedef bool (*tSVGASetupDriver)(const char*);

static tSVGASetupDriver svgaDriver[] = {
#if C_SVGA_S3
	&VGA_SetupS3Trio, // default
#endif
#if C_SVGA_TSENG
	&VGA_SetupET4K,
	&VGA_SetupNewET4K,
	&VGA_SetupET3K,
#endif
#if C_SVGA_PARADISE
	&VGA_SetupPVGA1A,
#endif
	&VGA_SetupSVGANone
};

// Driver initialization routine to use.
tSVGASetupDriver setup_chipset = NULL;

void VGA_SVGA_Init(Section * sec) {
	char chipset_lcase[50];
	Section_prop * section=static_cast<Section_prop *>(sec);
	const char* chipset;
	std::string cline;
	if (control->cmdline->FindString("-svgachipset",cline,false)) {
		chipset = cline.c_str();
	} else {
		chipset = section->Get_string("svgachipset");
	}

	if(chipset) {
		strncpy(chipset_lcase, chipset, 50);
		lowcase(chipset_lcase);
	}
	else
		chipset_lcase[0] = 0;

	vga.vmemsize = 0;

	for(int i=0; i<sizeof(svgaDriver)/sizeof(svgaDriver[0]); i++)
		if(svgaDriver[i](chipset_lcase)) {
			setup_chipset = svgaDriver[i];
			break;
		}
	if(!setup_chipset)
		setup_chipset = svgaDriver[0];

	if(vga.vmemsize == 0) {
		int msize;
		if (control->cmdline->FindInt("-videoram",msize,false)) {
			vga.vmemsize = msize;
		} else {
			vga.vmemsize = section->Get_int("videoram"); // this value is in Kbytes
		}

		if(vga.vmemsize < 10)
			vga.vmemsize *= 1024; // very low number specified, assuming Mbytes

		vga.vmemsize = ((vga.vmemsize/256)*256)*1024; // all existing chipsets support only 256K granularity

	}
	vga.svga.bank_size = 64*1024; // most common value
}

void VGA_SetupSVGA(void) {
	if (machine==MCH_VGA) {
        if(setup_chipset)
            setup_chipset(NULL);
	}
}
