/*
 *  Copyright (C) 2002-2007  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dosbox.h"
#include "setup.h"
#include "vga.h"
#include "inout.h"
#include "mem.h"

// Paradise PVGA1A implementation
// TODOs: Some extended registers are not implemented completely.
//        Dual banks are not implemented.
//        WD BIOS extensions (did anybody ever use it???).
//        Text modes 41h, 47h, 66h, 67h, 69h.
//        Odd memory model modes 59h, 5ah, 5bh

#define crtc(blah) vga.crtc.blah
#define seq(blah) vga.seq.blah
#define attr(blah) vga.attr.blah
#define gfx(blah) vga.gfx.blah

/////////////////////////////////////////////////////////////////////////

VideoModeBlock ModeList_PVGA1A[]={
/* mode  ,type     ,sw  ,sh  ,tw ,th ,cw,ch ,pt,pstart  ,plength,htot,vtot,hde,vde special flags */
{ 0x054  ,M_TEXT   ,1056 ,688, 132,43, 8, 9, 1, 0xB0000, 0x4000, 192, 720, 132, 688, 0 },
{ 0x055  ,M_TEXT   ,1056 ,400, 132,25, 8, 16,1, 0xB0000, 0x2000, 192, 449, 132, 400, 0 },
{ 0x056  ,M_TEXT   ,1056 ,688, 132,43, 8, 9, 1, 0xB0000, 0x4000, 192, 720, 132, 688, 0 },
{ 0x057  ,M_TEXT   ,1056 ,400, 132,25, 8, 16,1, 0xB0000, 0x2000, 192, 449, 132, 400, 0 },
{ 0x058  ,M_EGA    ,800 , 600, 100,37, 8, 16,1, 0xA0000, 0xA000, 128 ,663 ,100, 600, 0 },
{ 0x05E  ,M_LIN8   ,640 , 400, 80 ,25, 8, 16,1, 0xA0000, 0x10000,100 ,449 ,80 , 400, 0 },
{ 0x05F  ,M_LIN8   ,640 , 480, 80 ,30, 8, 16,1, 0xA0000, 0x10000,100 ,525 ,80 , 480, 0 },
};


typedef struct {
	Bitu PR0A;
	Bitu PR0B;
	Bitu PR1;
	Bitu PR2;
	Bitu PR3;
	Bitu PR4;
	Bitu PR5;

	inline bool locked() { return (PR5&7)!=5; }

} SVGA_PVGA1A_DATA;

static SVGA_PVGA1A_DATA pvga1a = { 0, 0, 0, 0, 0, 0, 0 };


static void bank_setup_pvga1a()
{
// Note: There is some inconsistency in available documentation. Most sources tell that PVGA1A used
//       only 7 bits of bank index (VGADOC and Ferraro agree on that) but also point that there are
//       implementations with 1M of RAM which is simply not possible with 7-bit banks. This implementation
//       assumes that the eighth bit was actually wired and could be used. This does not conflict with
//       anything and actually works in WHATVGA just fine.
	if(pvga1a.PR1 & 0x08)
	{
		// TODO: Dual bank function is not supported yet
		// TODO: Requirements are not compatible with vga_memory implementation.
	}
	else
	{
		// Single bank config is straightforward
		vga.svga.bank_read = vga.svga.bank_write = pvga1a.PR0A;
		vga.svga.bank_size = 4*1024;
		VGA_SetupHandlers();
	}
}

void write_p3cf_pvga1a(Bitu port,Bitu val,Bitu iolen) {
	if(pvga1a.locked() && gfx(index) >= 0x09 && gfx(index) <= 0x0e)
		return;

	switch (gfx(index)) {
	case 0x09:
		// Bank A, 4K granularity, not using bit 7
		// Maps to A800h-AFFFh if PR1 bit 3 set and 64k config B000h-BFFFh if 128k config. A000h-AFFFh otherwise.
		pvga1a.PR0A = val;
		bank_setup_pvga1a();
		break;
	case 0x0a:
		// Bank B, 4K granularity, not using bit 7
		// Maps to A000h-A7FFh if PR1 bit 3 set and 64k config, A000h-AFFFh if 128k
		pvga1a.PR0B = val;
		bank_setup_pvga1a();
		break;
	case 0x0b:
		// Memory size. We only allow to mess with bit 3 here (enable bank B) - this may break some detection schemes
		pvga1a.PR1 = (pvga1a.PR1 & ~0x08) | (val & 0x08);
		bank_setup_pvga1a();
		break;
	case 0x0c:
		// Video configuration
		// TODO: Figure out if there is anything worth implementing here.
		pvga1a.PR2 = val;
		break;
	case 0x0d:
		// CRT control. Bits 3-4 contain bits 16-17 of CRT start.
		// TODO: Implement bit 2 (CRT address doubling - this mechanism is present in other chipsets as well,
		// but not implemented in DosBox core)
		pvga1a.PR3 = val;
		vga.config.display_start = (vga.config.display_start & 0xffff) | ((val & 0x18)<<13);
		vga.config.cursor_start = (vga.config.cursor_start & 0xffff) | ((val & 0x18)<<13);
		break;
	case 0x0e:
		// Video control
		// TODO: Figure out if there is anything worth implementing here.
		pvga1a.PR4 = val;
		break;
	case 0x0f:
		// Enable extended registers
		pvga1a.PR5 = val;
		break;
	default:
		LOG(LOG_VGAMISC,LOG_NORMAL)("VGA:GFX:PVGA1A:Write to illegal index %2X", gfx(index));
	}
}

Bitu read_p3cf_pvga1a(Bitu port,Bitu iolen) {
	if(pvga1a.locked() && gfx(index) >= 0x09 && gfx(index) <= 0x0e)
		return 0x0;

	switch (gfx(index)) {
	case 0x09:
		return pvga1a.PR0A;
	case 0x0a:
		return pvga1a.PR0B;
	case 0x0b:
		return pvga1a.PR1;
	case 0x0c:
		return pvga1a.PR2;
	case 0x0d:
		return pvga1a.PR3;
	case 0x0e:
		return pvga1a.PR4;
	case 0x0f:
		return pvga1a.PR5;
	default:
		LOG(LOG_VGAMISC,LOG_NORMAL)("VGA:GFX:PVGA1A:Read from illegal index %2X", gfx(index));
	}

	return 0x0;
}



void SetupVideoModes_PVGA1A() {
	for(int mode=0; mode<sizeof(ModeList_PVGA1A)/sizeof(ModeList_PVGA1A[0]); mode++)
	{
		long vram_required = (long)ModeList_PVGA1A[mode].sheight*ModeList_PVGA1A[mode].swidth; // works for LIN8 modes
		switch(ModeList_PVGA1A[mode].type)
		{
		case M_EGA:
			vram_required /= 2;
			break;
		case M_TEXT:
			vram_required /= 64;
			break;
		}
		if(vram_required < (long)vga.vmemsize)
			ModeList.push_back(&ModeList_PVGA1A[mode]);
	}
}

void FinishSetMode_PVGA1A(Bitu /*crtc_base*/, VGA_ModeExtraData* modeData)
{
// Reset to single bank and set it to 0. May need to unlock first (DPaint locks on exit)
	IO_Write(0x3ce, 0x0f);
	Bitu oldlock = IO_Read(0x3cf);
	IO_Write(0x3cf, 0x05);
	IO_Write(0x3ce, 0x09);
	IO_Write(0x3cf, 0x00);
	IO_Write(0x3ce, 0x0a);
	IO_Write(0x3cf, 0x00);
	IO_Write(0x3ce, 0x0b);
	Bit8u val = IO_Read(0x3cf);
	IO_Write(0x3cf, val & ~0x08);
	IO_Write(0x3ce, 0x0c);
	IO_Write(0x3cf, 0x00);
	IO_Write(0x3ce, 0x0d);
	IO_Write(0x3cf, 0x00);
	IO_Write(0x3ce, 0x0e);
	IO_Write(0x3cf, 0x00);
	IO_Write(0x3ce, 0x0f);
	IO_Write(0x3cf, oldlock);

	if(vga.mode != M_VGA)
	{
        vga.config.compatible_chain4 = false; // in process of verification
		vga.vmemwrap = vga.vmemsize; // potential TODO -- do this through PR1
	}
	else
	{
        vga.config.compatible_chain4 = true;
		vga.vmemsize = 256*1024;
	}
	VGA_SetupHandlers();
}

void DetermineMode_PVGA1A()
{
	// Close replica from the base implementation. It will stay here
	// until I figure a way to either distinguish M_VGA and M_LIN8 or
	// merge them.
	if (vga.attr.mode_control & 1) {
		if (vga.gfx.mode & 0x40) VGA_SetMode((CurMode->mode<=0x13)?M_VGA:M_LIN8); // Ugly...
		else if (vga.gfx.mode & 0x20) VGA_SetMode(M_CGA4);
		else if ((vga.gfx.miscellaneous & 0x0c)==0x0c) VGA_SetMode(M_CGA2);
		else VGA_SetMode(M_EGA);
	} else {
		VGA_SetMode(M_TEXT);
	}
}

void SetClock_PVGA1A(Bitu which,Bitu target) {
	vga.clockFreq[which]=1000*target;
	VGA_StartResize();
}

Bitu GetClock_PVGA1A() {
	return vga.clockFreq[(vga.misc_output >> 2) & 3];
}

bool VGA_SetupPVGA1A(const char* testname) {
	if(testname)
		return strstr(testname, "pvga1a") != NULL;

	svgaCard = SVGA_ParadisePVGA1A;

	svga.write_p3cf = &write_p3cf_pvga1a;
	svga.read_p3cf = &read_p3cf_pvga1a;

    svga.setup_videomodes = &SetupVideoModes_PVGA1A;
    svga.set_video_mode = &FinishSetMode_PVGA1A;
	svga.determine_mode = &DetermineMode_PVGA1A;
	svga.set_clock = &SetClock_PVGA1A;
	svga.get_clock = &GetClock_PVGA1A;

	if(vga.vmemsize < 512*1024)
	{
		vga.vmemsize = 256*1024;
		pvga1a.PR1 = 1<<6;
	}
	else if(vga.vmemsize > 512*1024)
	{
		vga.vmemsize = 1024*1024;
		pvga1a.PR1 = 3<<6;
	}
	else
		pvga1a.PR1 = 2<<6;

	// Paradise ROM signature
	PhysPt rom_base=PhysMake(0xc000,0);
	phys_writeb(rom_base+0x007d,'V');
	phys_writeb(rom_base+0x007e,'G');
	phys_writeb(rom_base+0x007f,'A');
	phys_writeb(rom_base+0x0080,'=');

	IO_Write(0x3cf, 0x05); // Enable!

	return true;
}
